﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using UnityEngine;
using UnityEngine.PlayerLoop;
using UnityEngine.Tilemaps;
using UnityEngine.U2D;
using UnityEngine.UI;

//MinimapManager is a singleton class responsible for managing the minimap and providing an interaction API
public class MinimapManager : MonoBehaviour
{
    //Singleton
    private static MinimapManager _Instance;
    public static MinimapManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<MinimapManager>();
            }

            return _Instance;
        }
    }

    //Public properties
    public float Zoom;

    //Animation variables
    private float _ZoomAnimationDuration = 3000.0f;
    private float _ZoomAnimationTimeElapsed = 0.0f;
    private float _ZoomStart = 0.0f;
    private float _ZoomTarget = 0.0f;
    private bool _IsZoomedIn = true;

    #region Behaviour
    private GameObject _MinimapGameObject;
    private RectTransform _MinimapRectTransform;
    private float _MinimapRadius;
    private RawImage _MinimapImage;
    private GameObject _MinimapIconsGameObject;
    private Dictionary<string, MinimapIcon> _MinimapIcons;
    private TileBase _MinimapWhiteTile;
    private Camera _ReferenceCamera;
    #endregion

    #region Creation
    private const float _TextureOrthoZoom = 15.0f;
    private Size _SingleTextureSize;
    private Size _FullTextureSize;
    private int _XIterations;
    private int _YIterations;
    private float _CameraHeight;
    private float _CameraWidth;
    private float _XOverscan;
    private float _YOverscan;
    private List<Texture2D> _FullTextures = new List<Texture2D>();
    #endregion

    private bool _CreationSuccessful = false;

    /// <summary>
    /// Initializes the minimap by setting up its creation parameters
    /// </summary>
    /// <param name="minimapGameObject">The base parent minimap object</param>
    /// <param name="referenceCamera">The camera to use as reference when generating the minimap</param>
    public void Initialize(GameObject minimapGameObject, Camera referenceCamera)
    {
        _MinimapGameObject = minimapGameObject;
        _MinimapRectTransform = minimapGameObject.GetComponent<RectTransform>();
        _MinimapRadius = _MinimapRectTransform.rect.width / 2.0f;
        _MinimapImage = _MinimapGameObject.FindChild("Mask").FindChild("MinimapImage").GetComponent<RawImage>();
        _MinimapIconsGameObject = _MinimapGameObject.FindChild("Mask").FindChild("Icons");
        _MinimapIcons = GetIcons();
        _MinimapWhiteTile = Resources.Load("Palette Tiles/Minimap Tiles/white") as Tile;
        _ReferenceCamera = referenceCamera;
        Zoom = ConfigurationManager.Instance.Minimap.ZoomedIn;
        _ZoomStart = Zoom;
        _ZoomTarget = Zoom;
    }

    /// <summary>
    /// Creates the minimap image
    /// </summary>
    /// <returns>Was the creation successful or not?</returns>
    public bool CreateMinimap()
    {
        try
        {
            //Set up our reference camera for taking images of the city to generate the minimap
            _ReferenceCamera.targetTexture = Resources.Load("MainCameraRenderTexture") as RenderTexture;
            _SingleTextureSize = new Size(_ReferenceCamera.targetTexture.width, _ReferenceCamera.targetTexture.height);
            _ReferenceCamera.gameObject.GetComponent<PixelPerfectCamera>().enabled = false;    //We need to disable this temporarily so we can change ortho
            GameManager.Instance.PlayerCarGameObject.SetActive(false);  //Let's also hide the player car

            //Let's store our old properties to restore later
            RenderTexture oldActiveRenderTexture = RenderTexture.active;
            Vector3 oldReferencePosition = _ReferenceCamera.transform.position;
            bool wasCameraOrtho = _ReferenceCamera.orthographic;
            float oldOrthoSize = _ReferenceCamera.orthographicSize;
            int oldLayer = _ReferenceCamera.cullingMask;
            UnityEngine.Color oldColour = _ReferenceCamera.backgroundColor;

            //Now get the whole minimap as a texture
            RenderTexture.active = _ReferenceCamera.targetTexture;
            Texture2D minimapTexture = GetFullMinimapImageTexture();

            //Time to restore our properties
            _ReferenceCamera.targetTexture = null;
            _ReferenceCamera.transform.position = oldReferencePosition;
            _ReferenceCamera.orthographic = wasCameraOrtho;
            _ReferenceCamera.orthographicSize = oldOrthoSize;
            _ReferenceCamera.cullingMask = oldLayer;
            _ReferenceCamera.backgroundColor = oldColour;
            RenderTexture.active = oldActiveRenderTexture;
            _ReferenceCamera.gameObject.GetComponent<PixelPerfectCamera>().enabled = true;  //Now re-enable
            GameManager.Instance.PlayerCarGameObject.SetActive(true);  //Show the player car
            GameManager.Instance.MinimapTilemap.gameObject.SetActive(false);

            if (minimapTexture != null)
            {
                //We succeeded
                _MinimapImage.texture = minimapTexture;
                _CreationSuccessful = true;
                return true;
            }

            else
            {
                //No texture so we failed
                UnityEngine.Debug.LogError("ERROR: The minimap texture returned was null, returning false.");
                return false;
            }
        }

        catch (Exception ex)
        {
            UnityEngine.Debug.LogError("ERROR: Exception occurred when creating mini-map capture images. The exception is: " + ex);
            return false;
        }
    }

    public void Update()
    {
        if (_CreationSuccessful)
        {
            if(Input.GetKeyDown(KeyCode.Z))
            {
                //Animate the toggling of zoom
                if (!_IsZoomedIn)
                {
                    AnimateZoom(ConfigurationManager.Instance.Minimap.ZoomedOut, ConfigurationManager.Instance.Minimap.ZoomedIn, ConfigurationManager.Instance.Minimap.ZoomAnimationTime);
                    _IsZoomedIn = true;
                }

                else
                {
                    AnimateZoom(ConfigurationManager.Instance.Minimap.ZoomedIn, ConfigurationManager.Instance.Minimap.ZoomedOut, ConfigurationManager.Instance.Minimap.ZoomAnimationTime);
                    _IsZoomedIn = false;
                }
            }

            UpdateZoom();

            //0, 0 is the bottom left corner
            //U moves along the X axis
            //V goes from the bottom upwards

            //We get our position in our minimap image space. We need to subtract the origin, as we need to account for the fact the minimap image may not start at 0, 0
            float playerXCoord = (GameManager.Instance.PlayerCarGameObject.transform.position.x - GameManager.Instance.TerrainTilemap.origin.x) / GameManager.Instance.TerrainTilemap.size.x;
            playerXCoord = playerXCoord - (playerXCoord * _XOverscan);

            float playerYCoord = ((GameManager.Instance.PlayerCarGameObject.transform.position.y - GameManager.Instance.TerrainTilemap.origin.y) / GameManager.Instance.TerrainTilemap.size.y);
            playerYCoord = playerYCoord - (playerYCoord * _YOverscan);

            //Now normalize the position into UV space
            float normalizedPlayerXCoord = (playerXCoord - (Zoom / 2.0f));
            float normalizedPlayerYCoord = (playerYCoord - (Zoom / 2.0f));
            Mathf.Clamp(normalizedPlayerXCoord, 0.0f, 1.0f);
            Mathf.Clamp(normalizedPlayerYCoord, 0.0f, 1.0f);

            //Set the image's position to be correct based on the player's position
            _MinimapImage.uvRect = new Rect(normalizedPlayerXCoord, normalizedPlayerYCoord, Zoom, Zoom);

            foreach (KeyValuePair<string, MinimapIcon> iconKeyVal in _MinimapIcons)
            {
                //Let's update the icons now
                MinimapIcon icon = iconKeyVal.Value;
                bool shouldRenderIcon = true;

                //Let's make sure we don't render this icon if we're on a mission
                if ((icon.Flags & (int)MinimapIconFlags.NotOnMission) != 0)
                {
                    shouldRenderIcon = !MissionsManager.Instance.IsOnMission;
                }

                Vector3 iconWorldPos = icon.GetWorldPosition();
                if (shouldRenderIcon && iconWorldPos != Constants.SentinelVector3)
                {
                    icon.gameObject.SetActive(true);

                    //Let's get our icon into the same UV space as the map image
                    float iconXCoord = (iconWorldPos.x - GameManager.Instance.TerrainTilemap.origin.x) / GameManager.Instance.TerrainTilemap.size.x;
                    iconXCoord = iconXCoord - (iconXCoord * _XOverscan);

                    float iconYCoord = (iconWorldPos.y - GameManager.Instance.TerrainTilemap.origin.y) / GameManager.Instance.TerrainTilemap.size.y;
                    iconYCoord = iconYCoord - (iconYCoord * _YOverscan);

                    float normalizedIconXCoord = (iconXCoord - (Zoom / 2.0f));
                    float normalizedIconYCoord = (iconYCoord - (Zoom / 2.0f));

                    Vector2 iconPos = new Vector2(normalizedIconXCoord, normalizedIconYCoord);
                    Vector2 centrePos = new Vector2(normalizedPlayerXCoord, normalizedPlayerYCoord);

                    //Now we can normalize the difference (we do zoom / 2 to account for the centre of the minimap) and multiply by our radius to get into minimap UI space
                    Vector2 diff = ((iconPos - centrePos) * (1.0f / (Zoom / 2.0f))) * _MinimapRadius;

                    //Finally, clamp so we don't go outside the minimap circle and display
                    Vector2 clampedDiff = Vector2.ClampMagnitude(diff, _MinimapRadius);
                    icon.RectTransform.anchoredPosition = new Vector2(clampedDiff.x, clampedDiff.y);

                    //This icon should only appear in the local space if visible in the area that the minimap covers
                    if((icon.Flags & (int)MinimapIconFlags.LocalSpace) != 0)
                    {
                        shouldRenderIcon = !(diff.magnitude > _MinimapRadius);
                    }

                    //Update the icon rotation if needed to match the rotation of the reference object
                    if(icon.ReferenceGameObject != null && (icon.Flags & (int)MinimapIconFlags.MaintainRotation) != 0)
                    {
                        icon.RectTransform.rotation = icon.ReferenceGameObject.transform.rotation;
                    }
                }

                else
                {
                    shouldRenderIcon = false;
                }

                icon.gameObject.SetActive(shouldRenderIcon);
            }
        }
    }

    /// <summary>
    /// Draws a tile on the minimap layer used for image capture
    /// </summary>
    /// <param name="cellPosition">The position of the cell</param>
    /// <param name="colour">The colour of the cell</param>
    public void DrawMinimapTile(Vector3Int cellPosition, UnityEngine.Color colour)
    {
        GameManager.Instance.MinimapTilemap.SetTile(cellPosition, _MinimapWhiteTile);
        GameManager.Instance.MinimapTilemap.SetTileFlags(cellPosition, TileFlags.None);
        GameManager.Instance.MinimapTilemap.SetColor(cellPosition, colour);
    }

    /// <summary>
    /// Adds an icon to the minimap
    /// </summary>
    /// <param name="referenceObject">The reference game object that the icon represents</param>
    /// <param name="iconName">The name of the icon</param>
    /// <param name="tintColour">The colour to use for tinting the icon image</param>
    /// <param name="renderSize">The render size of the icon image</param>
    /// <param name="flags">The bitmasked flags of the icon</param>
    /// <param name="iconImageSprite">The sprite to use for the icon</param>
    public void AddIcon(GameObject referenceObject, string iconName, UnityEngine.Color tintColour, Vector2 renderSize, int flags = 0, Sprite iconImageSprite = null)
    {
        //Remove any duplicate
        RemoveIcon(iconName);

        //Now set the icon properties up
        GameObject iconObject = Instantiate(GameController.Instance.MinimapIconPrefab);
        iconObject.name = iconName;

        if (iconImageSprite != null)
        {
            iconObject.GetComponent<Image>().sprite = iconImageSprite;
        }

        iconObject.GetComponent<Image>().color = tintColour;
        iconObject.GetComponent<RectTransform>().sizeDelta = renderSize;

        iconObject.GetComponent<MinimapIcon>().IconName = iconName;
        iconObject.GetComponent<MinimapIcon>().GetPositionFromReferenceGameObject = true;
        iconObject.GetComponent<MinimapIcon>().ReferenceGameObject = referenceObject;
        iconObject.GetComponent<MinimapIcon>().RectTransform = iconObject.GetComponent<RectTransform>();
        iconObject.GetComponent<MinimapIcon>().Flags = flags;
        iconObject.transform.SetParent(_MinimapIconsGameObject.transform, false);

        //And store the icon
        _MinimapIcons[iconName] = iconObject.GetComponent<MinimapIcon>();
    }

    /// <summary>
    /// Adds an icon to the minimap
    /// </summary>
    /// <param name="worldPosition"></param>
    /// <param name="iconName">The name of the icon</param>
    /// <param name="tintColour">The colour to use for tinting the icon image</param>
    /// <param name="renderSize">The render size of the icon image</param>
    /// <param name="flags">The bitmasked flags of the icon</param>
    /// <param name="iconImageSprite">The sprite to use for the icon</param>
    public void AddIcon(Vector3 worldPosition, string iconName, UnityEngine.Color tintColour, Vector2 renderSize, int flags = 0, Sprite iconImageSprite = null)
    {
        //Remove any duplicate
        RemoveIcon(iconName);

        //Now set the icon properties up
        GameObject iconObject = Instantiate(GameController.Instance.MinimapIconPrefab);
        iconObject.name = iconName;

        if (iconImageSprite != null)
        {
            iconObject.GetComponent<Image>().sprite = iconImageSprite;
        }

        iconObject.GetComponent<Image>().color = tintColour;
        iconObject.GetComponent<RectTransform>().sizeDelta = renderSize;

        iconObject.GetComponent<MinimapIcon>().IconName = iconName;
        iconObject.GetComponent<MinimapIcon>().GetPositionFromReferenceGameObject = false;
        iconObject.GetComponent<MinimapIcon>().IconPosition = worldPosition;
        iconObject.GetComponent<MinimapIcon>().RectTransform = iconObject.GetComponent<RectTransform>();
        iconObject.GetComponent<MinimapIcon>().Flags = flags;
        iconObject.transform.SetParent(_MinimapIconsGameObject.transform, false);

        //And store the icon
        _MinimapIcons[iconName] = iconObject.GetComponent<MinimapIcon>();
    }

    /// <summary>
    /// Removes an icon with the icon name from the minimap
    /// </summary>
    /// <param name="iconName">The name of the icon to remove</param>
    public void RemoveIcon(string iconName)
    {
        if(_MinimapIcons.ContainsKey(iconName))
        {
            //If we've got an icon named, destory it and remove it
            Destroy(_MinimapIcons[iconName].RectTransform.gameObject);
            _MinimapIcons.Remove(iconName);
        }
    }

    /// <summary>
    /// Animates the zoom between a starting and target value over a duration
    /// </summary>
    /// <param name="zoomStart">The starting zoom value</param>
    /// <param name="zoomTarget">The target zoom value</param>
    /// <param name="duration">The duration of the animation</param>
    public void AnimateZoom(float zoomStart, float zoomTarget, float duration)
    {
        Zoom = zoomStart;
        _ZoomStart = zoomStart;
        _ZoomTarget = zoomTarget;
        _ZoomAnimationDuration = duration;
        _ZoomAnimationTimeElapsed = 0.0f;
    }

    /// <summary>
    /// Gets the full minimap texture of the city
    /// </summary>
    /// <returns>The full texture</returns>
    private Texture2D GetFullMinimapImageTexture()
    {
        try
        {
            //Final bits of setup
            _CameraHeight = _TextureOrthoZoom * 2.0f;
            _CameraWidth = _CameraHeight * _ReferenceCamera.aspect;
            _ReferenceCamera.orthographic = true;
            _ReferenceCamera.backgroundColor = UnityEngine.Color.clear;

            //Compute our heights and iterations
            _XIterations = Convert.ToInt32(Math.Ceiling(GameManager.Instance.TerrainTilemap.size.x / _CameraWidth));
            _YIterations = Convert.ToInt32(Math.Ceiling(GameManager.Instance.TerrainTilemap.size.y / _CameraHeight));
            _FullTextureSize = new Size(_SingleTextureSize.Width * _XIterations, _SingleTextureSize.Height * _YIterations);

            //Create our full textures
            if (CreateFullTextures())
            {
                //Now we combined them
                Texture2D fullMinimapImageTexture = new Texture2D(_SingleTextureSize.Width * _XIterations, _SingleTextureSize.Height * _YIterations);

                int totalIterations = 0;
                for (int x = 0; x < _SingleTextureSize.Width * _XIterations; x += _SingleTextureSize.Width)
                {
                    for (int y = 0; y < _SingleTextureSize.Height * _YIterations; y += _SingleTextureSize.Height)
                    {
                        fullMinimapImageTexture.SetPixels(x, y, _SingleTextureSize.Width, _SingleTextureSize.Height, _FullTextures[totalIterations].GetPixels());
                        fullMinimapImageTexture.Apply();
                        totalIterations++;
                    }
                }

                //Let's assume we have no overscan
                _XOverscan = 0.0f;
                _YOverscan = 0.0f;

                if (GameManager.Instance.TerrainTilemap.size.x % _CameraWidth != 0.0f)  //Check for overscan
                {
                    //We take our excess units and multiply it by the pixels per unit. We then take that overscan pixels value and multiply it by 1 / TotalTextureSize to convert into UVs
                    float excessXUnits = _CameraWidth - (GameManager.Instance.TerrainTilemap.size.x % _CameraWidth);
                    _XOverscan = (((_SingleTextureSize.Width) / _CameraWidth) * excessXUnits) * (1.0f / (_SingleTextureSize.Width * _XIterations));
                }

                if (GameManager.Instance.TerrainTilemap.size.y % _CameraHeight != 0.0f) //Check for overscan
                {
                    float excessYUnits = _CameraHeight - (GameManager.Instance.TerrainTilemap.size.y % _CameraHeight);
                    _YOverscan = (((_SingleTextureSize.Height) / _CameraHeight) * excessYUnits) * (1.0f / (_SingleTextureSize.Height * _YIterations));
                }

                //We can output the full texture here if needed
                return fullMinimapImageTexture;
            }

            else
            {
                UnityEngine.Debug.LogError("ERROR: Failed to perform mini-map detail pass, returning null.");
                return null;
            }
        }

        catch (Exception ex)
        {
            UnityEngine.Debug.LogError("ERROR: Exception occurred when getting full mini-map image texture. The exception is: " + ex);
            return null;
        }
    }

    /// <summary>
    /// Gets a texture of what the camera sees, tinted by the passed colour
    /// </summary>
    /// <param name="tintColour">The colour to tint the texture</param>
    /// <returns>The tinted texture</returns>
    private Texture2D GetCameraImageTextureTinted(UnityEngine.Color tintColour)
    {
        //Render the camera view to a texture
        _ReferenceCamera.Render();
        Texture2D cameraImageTexture = new Texture2D(_SingleTextureSize.Width, _SingleTextureSize.Height);
        cameraImageTexture.ReadPixels(new Rect(0, 0, _SingleTextureSize.Width, _SingleTextureSize.Height), 0, 0);
        cameraImageTexture.Apply();

        //Get all the default pixels, create a tinted list
        UnityEngine.Color[] colouredPixels = cameraImageTexture.GetPixels();
        List<UnityEngine.Color> tintedColours = new List<UnityEngine.Color>();

        foreach(UnityEngine.Color colouredPixel in colouredPixels)
        {
            //Compute the average, convert to grayscale and then tint
            float avg = (colouredPixel.r + colouredPixel.b + colouredPixel.g) / 3.0f;
            UnityEngine.Color grayscaleColour = new UnityEngine.Color(avg, avg, avg, 1.0f);
            UnityEngine.Color tintedColour = grayscaleColour * tintColour;

            if(tintedColour == UnityEngine.Color.black)
            {
                tintedColour = UnityEngine.Color.clear;
            }

            //Add the tinted colour
            tintedColours.Add(tintedColour);
        }

        //Replace the texture with the tinted values and return it
        cameraImageTexture.SetPixels(tintedColours.ToArray());
        cameraImageTexture.Apply();
        return cameraImageTexture;
    }

    /// <summary>
    /// Gets a texture of what the camera sees, filling any non-clear pixel with the passed colour
    /// </summary>
    /// <param name="solidColour">The colour to set non-clear pixels</param>
    /// <returns></returns>
    private Texture2D GetCameraImageTextureSolid(UnityEngine.Color solidColour)
    {
        //Render the camera view to a texture
        _ReferenceCamera.Render();
        Texture2D cameraImageTexture = new Texture2D(_SingleTextureSize.Width, _SingleTextureSize.Height);
        cameraImageTexture.ReadPixels(new Rect(0, 0, _SingleTextureSize.Width, _SingleTextureSize.Height), 0, 0);
        cameraImageTexture.Apply();

        //Get all the default pixels, create a tinted list
        UnityEngine.Color[] colouredPixels = cameraImageTexture.GetPixels();
        List<UnityEngine.Color> solidColours = new List<UnityEngine.Color>();

        foreach (UnityEngine.Color colouredPixel in colouredPixels)
        {
            if(colouredPixel != UnityEngine.Color.clear)
            {
                //Non-clear so let's add the solid colour
                solidColours.Add(solidColour);
            }

            else
            {
                //It's clear, let's ignore this
                solidColours.Add(UnityEngine.Color.clear);
            }
        }

        //Replace the texture with the solid colour values and return it
        cameraImageTexture.SetPixels(solidColours.ToArray());
        cameraImageTexture.Apply();
        return cameraImageTexture;
    }

    /// <summary>
    /// Creates the list of full textures that make up the entire minimap image
    /// </summary>
    /// <returns>Was the creation successful?</returns>
    private bool CreateFullTextures()
    {
        //Get our colours for each part
        UnityEngine.Color buildingColour = ColoursManager.Instance.Colours["MinimapBuilding"].Colour;
        UnityEngine.Color roadsColour = ColoursManager.Instance.Colours["MinimapRoad"].Colour;
        UnityEngine.Color terrainColour = ColoursManager.Instance.Colours["MinimapTerrain"].Colour;

        //Start at the bottom
        float startingX = GameManager.Instance.TerrainTilemap.origin.x + (_CameraWidth / 2.0f);
        float startingY = GameManager.Instance.TerrainTilemap.origin.y + (_CameraHeight / 2.0f);

        for (int x = 0; x < _XIterations; x++)
        {
            for (int y = 0; y < _YIterations; y++)
            {
                //Loop through as many times as we need to, move the camera to the current position
                _ReferenceCamera.transform.position = new Vector3((x * _CameraWidth) + startingX, (y * _CameraHeight) + startingY, _ReferenceCamera.transform.position.z);
                _ReferenceCamera.orthographicSize = _TextureOrthoZoom; //This HAS to go after changing position

                //Get the texture of just the buildings
                _ReferenceCamera.cullingMask = 1 << LayerMask.NameToLayer("Buildings");
                Texture2D buildingsImageTexture = GetCameraImageTextureTinted(buildingColour);

                //Get the texture of just the roads
                _ReferenceCamera.cullingMask = 1 << LayerMask.NameToLayer("Roads");
                Texture2D roadsImageTexture = GetCameraImageTextureSolid(roadsColour);

                //Create our list of colours
                List<UnityEngine.Color> buildingsColours = buildingsImageTexture.GetPixels().ToList();
                List<UnityEngine.Color> roadsColours = roadsImageTexture.GetPixels().ToList();
                List<UnityEngine.Color> finalColours = new List<UnityEngine.Color>();

                for(int i = 0; i < buildingsColours.Count; i++)
                {
                    //Building here, it takes priority, add it to the final list
                    if(buildingsColours[i] != UnityEngine.Color.clear)
                    {
                        finalColours.Add(buildingsColours[i]);
                    }

                    //No builidng but a road, add it to the final list
                    else if (roadsColours[i] != UnityEngine.Color.clear)
                    {
                        finalColours.Add(roadsColours[i]);
                    }

                    //No building or road so it must be terrain, add it to the final list
                    else
                    {
                        finalColours.Add(terrainColour);
                    }
                }

                //Got this area done, let's add it to the full textures and move onto the next area if we have one
                Texture2D combinedImageTexture = new Texture2D(buildingsImageTexture.width, buildingsImageTexture.height);
                combinedImageTexture.SetPixels(finalColours.ToArray());
                combinedImageTexture.Apply();
                _FullTextures.Add(combinedImageTexture);

                //We could save the sections here
            }
        }

        return true;
    }

    /// <summary>
    /// Gets the minimap icon components with their name as an ID
    /// </summary>
    /// <returns>Returns a dictionary of icons with a string ID and a MinimapIcon object value</returns>
    private Dictionary<string, MinimapIcon> GetIcons()
    {
        Dictionary<string, MinimapIcon> retDict = new Dictionary<string, MinimapIcon>();

        if (_MinimapIconsGameObject != null)
        {
            foreach (Transform childTransform in _MinimapIconsGameObject.transform.GetAllChildren())
            {
                MinimapIcon thisMinimapIcon = childTransform.gameObject.GetComponent<MinimapIcon>();

                if (thisMinimapIcon != null)
                {
                    retDict[childTransform.gameObject.name] = thisMinimapIcon;
                }
            }
        }

        return retDict;
    }

    /// <summary>
    /// Updates the zoom value based on the animation state
    /// </summary>
    private void UpdateZoom()
    {
        if(_ZoomAnimationTimeElapsed < _ZoomAnimationDuration)
        {
            //Still animating so update now
            _ZoomAnimationTimeElapsed += Time.deltaTime * 1000.0f;
            Zoom = Mathf.Lerp(_ZoomStart, _ZoomTarget, _ZoomAnimationTimeElapsed / _ZoomAnimationDuration);
        }

        else
        {
            //All done animating, snap to the final value
            Zoom = _ZoomTarget;
        }
    }
}
